import random
from typing import List
from jpype import JArray, JString
from prompta.core.env.grid_run_env import GridRunEnv
from prompta.core.language import BaseLanguage
from prompta.utils.java_libs import Word, DefaultQuery


class GridRunLanguage(BaseLanguage):

    def __init__(self, alphabet: List[str]=None) -> None:
        self.env = GridRunEnv()
        self._alphabet = ["0", "1", "2", "3", "4"]

    def in_language(self, input_str: str) -> bool:
        if len(input_str) == 0:
            return False
        if '1' in input_str:
            return False
        else:
            for i, s in enumerate(input_str):
                if s != '2':
                    continue
                if '4' not in s[i:]:
                    return False
            return input_str[-1] == '3'
        
    def counterexample(self, aut, _type=str):
        # TODO: Use random action to generate a number of trajectories, record all the tile type of the observations, then check if they are in the language by using in_language(input_str) and aut.isAccepting(input_str). If any of these result is unmatched, it is a counterexample.
        max_steps = 100
        num_trajectories = 1000

        for _ in range(num_trajectories):
            state = self.env.reset()
            trajectory = [state[2]]

            for _ in range(max_steps):
                if random.random() < 0.1:
                    action = random.randint(0, 3)
                else:
                    action = self.env.get_safe_action()
                next_state, _, _, _ = self.env.step(action)

                tile_type = next_state[2]
                trajectory.append(tile_type)

                char_list = [str(_) for _ in trajectory]
                trajectory_str = ''.join(char_list)
                word = Word.fromArray(JArray(JString)(char_list), 0, len(char_list))
                if aut.accepts(word) != self.in_language(trajectory_str):
                    ce = trajectory_str
                    print("Counterexample found:", DefaultQuery(word, self.in_language(trajectory_str)))

                    if _type == str:
                        return ce
                    return DefaultQuery(word, self.in_language(trajectory_str))
        return None

    @property
    def alphabet(self):
        return self._alphabet
    
    @property
    def definition(self):
        return """A robot is operating in a grid world and can visit four types of tiles: red, yellow, blue, green. They correspond to lava (red), recharging (yellow), water (blue), and drying (green) tiles. The robot is to visit tiles according to some set of rules. This will be
        recorded as a sequence of colors.
        Rules include:
        1. The sequence must contain at least one yellow tile, i.e., eventually recharge.
        2. The sequence must not contain any red tiles, i.e., lava must be avoided at all costs.
        3. If blue is visited, then you must visit green *before* yellow, i.e., the robot must dry off before recharging.
        A positive example must conform to all rules. Further note that repeated sequential colors can be replaced with a single instance."""
    
    @property
    def examples(self):
        return {
            'pos': {
                'query': """Given a sequence of colors,         • [yellow,blue,green,yellow], does this sequence belongs to the language?""",
                'answer': str({'reason': 'it dries itself before going to recharging.', 'answer': True})
            },
            'neg': {
                'query': """Given a sequence of colors,         • [yellow,blue,green,red], does this sequence belongs to the language?""",
                'answer': str({'reason': 'It reaches a lava tile.', 'answer': False})
            }
        }

    @property
    def context_name(self):
        return type(self.env).__name__
    
    @property
    def name(self):
        return type(self).__name__
